/*
 *  Copyright 2002-2015 Barcelona Supercomputing Center (www.bsc.es)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package matmul.files;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.StringTokenizer;


public class Matmul {
	private static int MSIZE;
	private static int BSIZE;

	private static String[][] AfileNames;
	private static String[][] BfileNames;
	private static String[][] CfileNames;

    private static String inputFolder;
    private static String outputFolder;

	
	
	private static void usage() {
		System.out.println("    Usage: matmul.files.Matmul <MSize> <BSize>");
	}
	
	public static void main(String[] args) throws Exception {
		// Check and get parameters
		if (args.length != 2) {
			usage();
			throw new Exception("[ERROR] Incorrect number of parameters");
		}
		inputFolder = args[0];
		outputFolder = args[1];

        MSIZE = 6;
        BSIZE = 8;
		
		// Initialize matrices
		System.out.println("[LOG] MSIZE parameter value = " + MSIZE);
		System.out.println("[LOG] BSIZE parameter value = " + BSIZE);
		initializeVariables(inputFolder, outputFolder);
		// initializeMatrix(AfileNames, 2);
		// initializeMatrix(BfileNames, 3);
		initializeMatrix(CfileNames, 0);
		
		// Compute matrix multiplication C = A x B
		computeMultiplication();
		
		// Uncomment the following line if you wish to see the result in the stdout
		//printMatrix(CfileNames, "C (Result)");
		// Uncomment the following line if you wish to store the result in a file
		//storeMatrix("c_result.txt");
		
		// End
		System.out.println("[LOG] Main program finished.");
	}
	
	private static void initializeVariables (String inputFolder, String outputFolder) {
		AfileNames = new String[MSIZE][MSIZE];
		BfileNames = new String[MSIZE][MSIZE];
		CfileNames = new String[MSIZE][MSIZE];
		for ( int i = 0; i < MSIZE; i ++ ) {
			for ( int j = 0; j < MSIZE; j ++ ) {
				AfileNames[i][j] = inputFolder + "A." + i + "." + j;
				BfileNames[i][j] = inputFolder + "B." + i + "." + j;
				CfileNames[i][j] = outputFolder + "C." + i + "." + j;
			}
		}
	}
	
	private static void initializeMatrix(String[][] fileNames, int initVal) throws Exception {
		for (int i = 0; i < MSIZE; ++i) {
			for (int j = 0; j < MSIZE; ++j) {
				Block.initBlockFile(fileNames[i][j], BSIZE, initVal);
			}
		}
	}
	
	private static void computeMultiplication() {
		System.out.println("[LOG] Computing result");
		for (int i = 0; i < MSIZE; i++) {
			for (int j = 0; j < MSIZE; j++) {
				for (int k = 0; k < MSIZE; k++) {
					MatmulImpl.multiplyAccumulative(CfileNames[i][j], AfileNames[i][k], BfileNames[k][j], BSIZE);
				}
            }
		}
	}
		
	@SuppressWarnings("unused")
	private static void storeMatrix (String fileName) throws Exception {
		try {
			FileOutputStream fos = new FileOutputStream(fileName);
			for (int i = 0; i < MSIZE; ++i) {
				for (int j = 0; j < MSIZE; ++j) {
					FileReader filereader = new FileReader(CfileNames[i][j]);
					BufferedReader br = new BufferedReader(filereader);
					StringTokenizer tokens;
					String nextLine;
					for (int iblock = 0; iblock < BSIZE; ++iblock) {
						nextLine = br.readLine();
						tokens = new StringTokenizer(nextLine);
						for (int jblock = 0; jblock < BSIZE && tokens.hasMoreTokens(); ++jblock) {
							String value = tokens.nextToken() + " ";
							fos.write(value.getBytes());
						}
					}
					fos.write("\n".getBytes());
					br.close();
					filereader.close();
				}
				fos.write("\n".getBytes());
			}
			fos.close();
		} catch (FileNotFoundException fnfe) {
			throw new Exception("[ERROR] Error storing result matrix", fnfe);
		} catch (IOException ioe) {
			throw new Exception("[ERROR] Error storing result matrix", ioe);
		}
	}

	@SuppressWarnings("unused")
	private static void printMatrix(String[][] fileNames, String name) throws Exception {
		System.out.println("MATRIX " + name);
		for (int i = 0; i < MSIZE; i++) {
			 for (int j = 0; j < MSIZE; j++) {
				Block aux = new Block(fileNames[i][j], BSIZE);
				aux.printBlock();
			 }
			 System.out.println("");
		 }
	}
	
}

