#!/bin/env python

import sys
import csv
import subprocess
import argparse
from collections import defaultdict


csv.field_size_limit(sys.maxsize)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("-pgcpp_exe", help="The executable to be run: Either 'protgraphtraverseintdryrun' or 'protgraphtraversefloatdryrun' (depending which of those are faster on the architecture)")
    parser.add_argument("-num_processes", help="The number of processes. NOTE: Use only #procs/2 many, since each query on a protein is run in this specific version on 2 threads!", type=int)
    parser.add_argument("-protein_graphs_bpcsr", help="The ProteinGraphs in BPCSR generated by ProtGraph")
    parser.add_argument("-out_detailed_statistics", help="Returns a detailed list for each protein, where we updated the max variants which is queryable in a timelimit")
    parser.add_argument("-out_limits", help="Returns a compact format containing the number of bins, bins itself and the maximum variants per bin for each proteins (to be added in protgraphcpp)")

    parser.add_argument("-max_precursor_da", help="The maximum Query which we allow. Average Aminoacid mass is ~110Da, for 60 Aminoacids it would be ~6600 which should cover almost everything. Defaults to 5000", default=5000, type=float)
    parser.add_argument("-bins", help="The number of bins. E.G.: Setting it to 32 results for a max_precursor_da of 5000 in (5000/32=)~156Da large windows. Defaults to 128", default=128, type=int)
    parser.add_argument("-ppm", help="The same ppm, which should or is used in the search engine. Defaults to 5 ppm", default=5, type=float)
    parser.add_argument("-timeout", help="The timeout, how long we want to wait for a query on a protein (in seconds). NOTE: In a dataset we have around 10 000 distinct queries (sometimes even more), so this decides how long the final FASTA-Generation will take, which sets the max variants limits appropiately!, Defaults to 5 seconds (which seems like agood limit)", default=5, type=float)
    parser.add_argument("-proteins_to_limit", help="'__all__'  (default) or '__none__' or as a comma seperated list 'P04637,P66871' This parameter limits a protein iff it timeouts", default="__all__")
    parser.add_argument("-max_limit", help="Limits the number of variants UP TO X. NOTE, if we find with the binary search, that we can only use less variants to query, we will use less variants. E.G. P53 with actual limits -1,-1,3,1,1,1 would be 5,5,3,1,1,1. Defaults to 5 (to only allow up to 5 variants)", default=5, type=int)
    parser.add_argument("-apply_smooting_method", help="The smoothin method (across a 3-element-sized window). Currently only median (default) or none.", default="median")
    return parser.parse_args()


def apply_smooting(array, method="median"):
    """ Array as input containing strings of numbers. Returns the smoothed array. """
    if method == "none":
        return array

    elif method == "median":
        # Get left and right limits
        left = max(array)
        right = min(array)

        # Apply median on left
        new_array = [
            sorted([left, array[0], array[1]])[1]
        ]
        # Apply median on middle
        for i in range(1, len(array) - 1):
            new_array.append(
                sorted([array[i-1], array[i], array[i+1]])[1]
            )

        # Apply median on right
        new_array.append(
                sorted([array[-2], array[-1], right])[1]
            )
        
        return [str(x) for x in new_array]

    else:
        raise Exception("Smooting not applicable. {} is not implemented.".format(method))


def error_check_get_index(tuple_a, list_tuple_b, tolerance=2):
    """ Checks if the tuple on [0] is exactly the same and [1] is the same with a 10^tolerance """
    for idx, tuple_b in enumerate(list_tuple_b):
        if tuple_a[0] == tuple_b[0] and tuple_a[1][:-tolerance] == tuple_b[1][:-tolerance]:
            return idx
    return -1
        

def error_check_get_index_list(element_a, list_b, tolerance=2):
    """ Checks if the elements is the same with a 10^tolerance """
    for idx, element_b in enumerate(list_b):
        if element_a[:-tolerance] == element_b[:-tolerance]:
            return idx
    return -1
        

def run_pgcpp(num_vars: str, input_query: str, output_statistics: str, args):
    """
    Wrapper around the subcall, allowing to run the binary with a custom num_vars to retrieve statistics
    """
    subprocess.call([
        args.pgcpp_exe,  # Executable
        args.protein_graphs_bpcsr,  # INPUT-BPCSR
        input_query,  # INPUT-Queries
        str(args.num_processes),  # INPUT-Num-Procs
        output_statistics,  # OUTPUT-Statistics
        num_vars,  # INPUT-Variants
        str(args.timeout),  # INPUT-Timeout
    ])


if __name__ == "__main__":
    # Parse arguments
    args = parse_args()

    # Generate n-many bins depending on max-query and save the initial query as csv
    query_bins = [(args.max_precursor_da/args.bins)*i for i in range(1,args.bins+1)]
    query_tuples = [(da - (da / 1000000) * args.ppm, da + (da / 1000000) * args.ppm) for da in query_bins]
    queries_identifiers = [str(int(x[0]* 1000000000)) for x in query_tuples]
    with open("temp_output.csv", "w") as out_file:
        csv_out = csv.writer(out_file)
        csv_out.writerows(query_tuples)

    # Start the very first (initial) run with infinite many variants
    run_pgcpp("-1", "temp_output.csv", "temp_output_runs.csv", args) # Get Initial first data

    # Go through the results of the initial run
    # We save them here:
    # results_dict["protein"]["query"] --> List of [Runtime, Number of paths, Used Variants] (last entry in list is most up to date)
    results_dict = defaultdict(defaultdict)
    timeout_proteins = []  # Already capture proteins/queries which timed out
    with open("temp_output_runs.csv") as pg_out:
        pg_out_csv = csv.reader(pg_out)

        for line in pg_out_csv:
            # Overview of indexes and their header names
            # line[0]   # Protein
            # line[1/2] # lower / upper
            # line[3]   # Number of Variants used
            # line[4]   # time in microseconds
            # line[5]   # number of found paths
            if not line[0] in results_dict:
                results_dict[line[0]] = defaultdict(list)

            results_dict[line[0]][line[1]].append(
                (
                    line[4], # Runtime
                    line[5], # Number of found paths
                    line[3], # Number of Variants used, 255 --> -1
                )
            )

            # Check if it is a timed out protein
            if int(line[4]) + 1 > args.timeout * 1000000:
                q = queries_identifiers[error_check_get_index_list(line[1], queries_identifiers)] # Get Query with error tolerance
                if args.max_limit == -1:
                    timeout_proteins.append(
                        # (protein, query), lower_limit, next_try, upper_limit
                        [(line[0],  q), 0, int(255/2), 255, (line[1], line[2])]                    
                    )
                else: 
                    timeout_proteins.append(
                        # (protein, query), lower_limit, next_try, upper_limit
                        [(line[0],  q), 0, int(args.max_limit/2), args.max_limit, (line[1], line[2])]                    
                    )

    # Do binary search: 
    print("Starting binary search to retrieve max_variants")
    while len(timeout_proteins) != 0:
        prot_query_ids = [x[0] for x in timeout_proteins]

        # Get the most occuring element (we will search them at once)
        check_vars = [x[2] for x in timeout_proteins]
        most_occuring_var = max(set(check_vars), key=check_vars.count)

        # Create Query to only search for most_occuring_var
        with open("temp_output.csv", "w") as out_file:
            csv_out = csv.writer(out_file)
            csv_out.writerows(
                # Only running the targeted query!
                set([(int(x[-1][0])/1000000000, int(x[-1][1])/1000000000) for x in timeout_proteins if x[2] == most_occuring_var])
            )
        
        # Get new Results
        run_pgcpp(str(most_occuring_var), "temp_output.csv", "temp_output_runs.csv", args)

        # Parse New Results and update timed out proteins
        with open("temp_output_runs.csv") as pg_out:
            pg_out_csv = csv.reader(pg_out)
            for line in pg_out_csv:
                idx = error_check_get_index((line[0], line[1]), prot_query_ids)
                if idx != -1:
                    if most_occuring_var == timeout_proteins[idx][2]:
                        print("Updating runtime for {} (query: {})  (for {} variants, needed {} microsecs)".format(
                            line[0], line[1], line[3], line[4]
                        ))
                        timout_entry = timeout_proteins.pop(idx)
                        prot_query_ids = [x[0] for x in timeout_proteins]

                        # Check if timeout
                        if int(line[4]) + 1 > args.timeout * 1000000:
                            # yes timeout, check less variants
                            # We schedule the next run
                            if round(((timout_entry[2] + timout_entry[1]) / 2)) != timout_entry[2]:
                                timeout_proteins.append([
                                    timout_entry[0], 
                                    timout_entry[1], 
                                    round(((timout_entry[2] + timout_entry[1]) / 2)), # Set new var check in half of lower and middle
                                    most_occuring_var,
                                    timout_entry[-1]
                                ])

                            # Special Case, if we even timeout out at 0 variants
                            if timout_entry[2] == 0 and timout_entry[1] == 0:
                                results_dict[timout_entry[0][0]][timout_entry[0][1]].append(
                                        (
                                            line[4], # Runtime
                                            line[5], # Number of found paths
                                            line[3], # Number of Variants used, 255 --> -1
                                        )
                                    )
                        else:
                            # No timeout, we can increase variants
                            # First, we write the entry into results_dict, this is the lowest max_vars we can search with (currently)
                            results_dict[timout_entry[0][0]][timout_entry[0][1]].append(
                                    (
                                        line[4], # Runtime
                                        line[5], # Number of found paths
                                        line[3], # Number of Variants used, 255 --> -1
                                    )
                                )
                            # Schedule next run:
                            if round(((timout_entry[2] + timout_entry[3]) / 2)) != timout_entry[2]:
                                timeout_proteins.append([
                                        timout_entry[0], 
                                        most_occuring_var, 
                                        round(((timout_entry[2] + timout_entry[3]) / 2)), # Set new var check in half of higher and middle
                                        timout_entry[3],
                                        timout_entry[-1]
                                    ])

    # Generate output statistics
    limits = defaultdict(list)

    # Detailed Output
    with open(args.out_detailed_statistics, "w") as out_detail:
        csv_out_detail = csv.writer(out_detail)
        csv_out_detail.writerow([
            "Protein", "Query_lower", "time_micros", "num_paths", "num_max_variants",
            "time_micros_updated_1", "num_paths_updated_1", "num_max_variants_updated_1", # For first update of lowest possible variants
            "time_micros_updated_2", "num_paths_updated_2", "num_max_variants_updated_2", # For second update of lowest possible variants
            # ... this would be then repeated. We stop prociding headers after the second update
        ])
        for key, val in results_dict.items():
            for key_query, val_time in val.items():
                csv_out_detail.writerow([
                    key, key_query, *[x for tup in val_time for x in tup]
                ])
                limits[key].append((key_query, val_time[-1][2]))

    # Protgraphcpp Output (here also the smoothing is applied)
    with open(args.out_limits, "w") as out_limits_cpp:
        csv_out_limits_cpp = csv.writer(out_limits_cpp)

        csv_out_limits_cpp.writerow(["#bins",args.bins])
        csv_out_limits_cpp.writerow(["bins",*query_bins])

        for key, val in limits.items(): 
            if key in args.proteins_to_limit or "__all__" in args.proteins_to_limit:
                csv_out_limits_cpp.writerow(
                    [
                        key, 
                        *apply_smooting(
                            [str(args.max_limit) if (int(y[1]) > int(args.max_limit)) or int(y[1]) == -1 else y[1] for y in sorted(val, key=lambda x: int(x[0]))],
                        method=args.apply_smooting_method
                        )
                    ]
                )

            else:
                csv_out_limits_cpp.writerow(
                    [
                        key, 
                        *apply_smooting(
                            [y[1] for y in sorted(val, key=lambda x: int(x[0]))],
                            method=args.apply_smooting_method
                        )
                    ]
                )
